//	Maze4DModel.swift
//
//	© 2025 by Jeff Weeks
//	See TermsOfUse.txt

import Foundation


let min4DShearFactor = 0.125	//	must be greater than zero
let max4DShearFactor = 2.0 * sqrt(2.0)
let default4DShearFactor = 1.375 * sqrt(2.0)	//	√2 => tubes are tangent

let coastingDeceleration: Double = 8.0	//	in tube lengths per second-squared

//	Enable gMaze4DForTalks only briefly,
//	when compiling 4D Maze for use on macOS during talks,
//	and then disable it again.
//#warning("disable gMaze4DForTalks")
let gMaze4DForTalks = false


@Observable class Maze4DModel: GeometryGamesUpdatable {

	var itsMaze: MazeData? = nil {
		didSet {
			itsGameIsOver = false
			changeCount += 1
		}
	}

	var itsOrientation: simd_quatd = gQuaternionIdentity {
		didSet {
			changeCount += 1			
		}
	}
	
	//	Assuming a constant frame rate, with gFramePeriod seconds
	//	from the start of one frame to the start of the next,
	//	let itsIncrement be the small additional rotation
	//	that we want post-multiply for each successive frame.
	//
	var itsIncrement: simd_quatd?	//	= nil when maze isn't rotating

	//	To distinguish nodes whose x-, y- and z-coordinates all agree,
	//	we slightly shear the 4D space before projecting down to 3D.
	//
	//	its4DShearFactor determines the relative spacing between tubes whose
	//	positions differ only in the ana-kata dimension.  The UI lets the player
	//	vary its4DShearFactor, so s/he can make it larger for good visibility,
	//	or make it smaller to better understand that parallel edges differing
	//	only in the ana-kata dimension really should appear superimposed in 3D.
	//	A reasonable default value is √2.  Of course its4DShearFactor should
	//	never be exactly 0, partly to avoid possible division by 0, but mainly
	//	to avoid the z-fighting that would occur if we tried to superimpose
	//	two tubes exactly.
	//
	var its4DShearFactor: Double =
		(gMaze4DForTalks ?
			min4DShearFactor :
			gMakeScreenshots ?
				1.0 :
				default4DShearFactor)
	
	var itsGameIsOver: Bool = false {
		didSet {
			if itsGameIsOver {
				Task { enqueueSoundRequest("MazeSolved.m4a") }
			}
			changeCount += 1
		}
	}
	
	//	To help new users understand that they should drag the slider,
	//	we'll let the slider blink when users launch the app.
	//	But as soon as the user touches the slider it will stop
	//	blinking and won't blink again until the user quits the app
	//	and restarts it.  At the moment it stops blinking, the goal
	//	will flash briefly to let the user know where to drag
	//	the slider too.
	//
	//	As a practical matter, the following two variables will
	//	record when the slider started blinking and when the goal
	//	began its flash.  When they aren't blinking or flashing,
	//	these variables will be nil.
	var itsSliderBlinkStartTime: CFAbsoluteTime?
		= (gMakeScreenshots ?
			nil :						//	Doesn't blink for screenshots.
			CFAbsoluteTimeGetCurrent()	//	Blinks at launch.
		  )
	var itsGoalFlashStartTime: CFAbsoluteTime? = nil {	//	doesn't flash until later
		didSet {
			changeCount += 1
		}
	}

	var changeCount: UInt64 = 0
	func updateModel() -> UInt64 {

		//	Update the maze's rotation.
		if let theIncrement = itsIncrement {

			//	Post-multiply ( => left-multiply) itsOrientation by theIncrement.
			itsOrientation = theIncrement * itsOrientation

			//	Normalize itsOrientation to keep its length
			//	from gradually drifting away from 1.
			//
			//		Note:  The length stays very close to 1
			//		at all times, so we don't have to worry about
			//		passing in the zero quaternion here.
			//
			itsOrientation = simd_normalize(itsOrientation)

			changeCount += 1
		}
		
		//	If the slider is coasting, update its position.
		if let theCoastingSpeed = itsMaze?.sliderCoastingSpeed,
		   let theSliderPosition = itsMaze?.sliderPosition {
		
			switch theSliderPosition {
			
			case .atNode(nodeIndex: _):
				assertionFailure("updateModel() received a slider with a non-nil coasting speed at a node")
				itsMaze?.sliderCoastingSpeed = nil
				
			case .onOutboundEdge(baseNodeIndex: let baseNodeIndex, direction: let direction, distance: let distance):

				var farNodeIndex = baseNodeIndex
				farNodeIndex[direction] += 1

				let theIncrement = gFramePeriod * theCoastingSpeed
				let theNewDistance = distance + theIncrement

				if theNewDistance <= 0.0 {
				
					itsMaze?.sliderPosition = .atNode(nodeIndex: baseNodeIndex)
					itsMaze?.sliderCoastingSpeed = nil

				} else if theNewDistance > 1.0 {
				
					itsMaze?.sliderPosition = .atNode(nodeIndex: farNodeIndex)
					itsMaze?.sliderCoastingSpeed = nil
						
				} else {
				
					itsMaze?.sliderPosition = .onOutboundEdge(
												baseNodeIndex: baseNodeIndex,
												direction: direction,
												distance: theNewDistance)

					//	Decelerate for next time.
					let Δspeed = coastingDeceleration * gFramePeriod
					if theCoastingSpeed > 0.0 {
						let theNewCoastingSpeed = theCoastingSpeed - Δspeed
						itsMaze?.sliderCoastingSpeed =
							theNewCoastingSpeed > 0.0 ?
							theNewCoastingSpeed :
							nil
					} else {
						let theNewCoastingSpeed = theCoastingSpeed + Δspeed
						itsMaze?.sliderCoastingSpeed =
							theNewCoastingSpeed < 0.0 ?
							theNewCoastingSpeed :
							nil
					}
				}
			}

			if sliderHasReachedGoal(maze: itsMaze) {
				itsGameIsOver = true
			}
		}
		
		//	Redraw the scene continuously while the slider is blinking.
		if itsSliderBlinkStartTime != nil {
			changeCount += 1
		}
		
		//	We don't need to continuously redraw the highlighted goal,
		//	but we do need to un-highlight it after a fraction of a second.
		if let theGoalFlashStartTime = itsGoalFlashStartTime {
			let theElapsedTime = CFAbsoluteTimeGetCurrent() - theGoalFlashStartTime
			let theFlashDuration = 0.25	//	seconds
			if theElapsedTime > theFlashDuration {
				itsGoalFlashStartTime = nil
			}
		}

		return changeCount
	}
}
